/* ==================================================================
   Copyright:   Travellers Tales (UK)
   File:        GhkFacet.inl
   Author:      Chris Halliday
   Date:        10th Nov 2006
   Format:      Generic C++
   Description: Single facet for Ghk algorithm
	
   $Id: GhkFacet.inl,v 1.5 2006-11-14 16:32:44 dconnell Exp $
   $Log: not supported by cvs2svn $
   Revision 1.4  2006/11/10 13:47:48  abrown
   Added CVS header


===================================================================== */

#ifndef GHKFACET_INL
#define GHKFACET_INL

#include "GhkFacet.h"

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
inline void GhkFacet::Init(int A, int B, int C) 
{
	Indices[0] = A; 
	Indices[1] = B; 
	Indices[2] = C;
	Obsolete = false;
}

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
inline bool GhkFacet::ClosestIsInternal()
{ 
	return Lambda1 >= 0.0f && Lambda2 >= 0.0f && Lambda1 + Lambda2 <= Det;
} 

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
inline bool GhkFacet::ContainsPoint( int PointIndex, VuVec Buffer[] )
{
	return  Buffer[ PointIndex ] == Buffer[ Indices[0] ] ||
			Buffer[ PointIndex ] == Buffer[ Indices[1] ] ||
			Buffer[ PointIndex ] == Buffer[ Indices[2] ];
}

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
inline void GhkFacet::GetClosestPoint( VuVec Buffer[], VuVec &Point )  
{
	VuVec &P0 = Buffer[ Indices[0] ];
	VuVec  V0 = Buffer[ Indices[1] ] - P0;
	VuVec  V1 = Buffer[ Indices[2] ] - P0;
	Point = P0 + ( (V0 * Lambda1 + V1 * Lambda2) / Det );
}

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
inline bool GhkFacet::link(int edge0, GhkFacet *facet, int edge1) 
{
    AdjacentFacets[edge0] = facet;
    AdjacentEdges[edge0] = edge1;
    facet->AdjacentFacets[edge1] = this;
    facet->AdjacentEdges[edge1] = edge0;

    bool b = Indices[edge0] == facet->Indices[ (edge1+1) % 3] &&
	Indices[ (edge0+1) % 3 ] == facet->Indices[edge1];
    return b;
}

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
inline bool GhkFacet::ComputeClosestPoint( VuVec *verts)
{
    VuVec& p0 = verts[Indices[0]]; 

    VuVec v1 = verts[Indices[1]] - p0;
    VuVec v2 = verts[Indices[2]] - p0;
    float v1dv1 = v1.MagnitudeSqr();
    float v1dv2 = v1.DotProduct(v2);
    float v2dv2 = v2.MagnitudeSqr();
    float p0dv1 = p0.DotProduct(v1); 
    float p0dv2 = p0.DotProduct(v2);
    
    Det = v1dv1 * v2dv2 - v1dv2 * v1dv2;
    Lambda1 = p0dv2 * v1dv2 - p0dv1 * v2dv2;
    Lambda2 = p0dv1 * v1dv2 - p0dv2 * v1dv1; 
    
    if (Det > 0.0f) 
	{	
		ClosestPoint = p0 + (Lambda1 * v1 + Lambda2 * v2) / Det;
		SqrDist = ClosestPoint.MagnitudeSqr();
		return true;
    }
    
    return false;
} 

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
inline void GhkFacet::Silhouette( VuVec& w, GhkEdgeBuffer& edgeBuffer) 
{
	Obsolete = true;
	edgeBuffer.Reset();
	AdjacentFacets[0]->Silhouette(AdjacentEdges[0], w, edgeBuffer);
	AdjacentFacets[1]->Silhouette(AdjacentEdges[1], w, edgeBuffer);
	AdjacentFacets[2]->Silhouette(AdjacentEdges[2], w, edgeBuffer);
}

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
inline void GhkFacet::Silhouette(int index,  VuVec& w, GhkEdgeBuffer& edgeBuffer) 
{
    if (!Obsolete) 
	{
		if ( ClosestPoint.DotProduct(w) < SqrDist) 
		{
			GhkEdge &Edge = edgeBuffer.Insert();
			Edge.Facet = this;
			Edge.Index = index;
		}	
		else 
		{
			Obsolete = true;
			int next = (index+1) % 3;
			AdjacentFacets[next]->Silhouette(AdjacentEdges[next], w, edgeBuffer);
			next = (next+1) % 3;
			AdjacentFacets[next]->Silhouette(AdjacentEdges[next], w, edgeBuffer);
		}
    }
}

#endif
